/* ----------------------------------------------------------------------
 *  awkos_pgtable.h defines the essential MACROS for processing page 
 *  table entries or page directory entries.
 *  
 *  NOTE!!
 *
 *  1. When a macro begins with an "_", the macro is not supposed to be used
 *     directly. Please use any macros which does not begin with "_"
 *  2. when a macro is called xxx_offset, the macro return an address.
 *  
 *  One logical (linear) address (32bits) consists of three parts
 *       p1          p2      offset
 *  +----------+----------+------------+
 *  |0000000000|0000000000|000000000000|
 *  +----------+----------+------------+
 *  HIGH                              LOW
 *
 *  One page table/page directory entry occupies 4 bytes 
 *          20 bits          12 bits
 *  +--------------------+------------+
 *  |00000000000000000000|00000D0000RP|
 *  +---------------------------------+
 *  HIGH 	                           LOW
 *  P: present bit
 *  R: read/write bit
 *  D: dirty bit
 *  ----------------------------------------------------------------------
 */

/* PAGE_SHIFT determines the page size */
#define PAGE_SHIFT	12
#define PAGE_SIZE	(4096)
#define PAGE_MASK	(0xFFFFFC00)
#define PAGE_ALIGN(addr)	(((addr)+PAGE_SIZE-1)&PAGE_MASK)
#define __PAGE_OFFSET		(0xC0000000)

#define PGDIR_SHIFT 22
#define PTRS_PER_PGD 1024
#define PTRS_PER_PTE 1024

#define _PAGE_PRESENT	0x001
#define _PAGE_RW	0x002
#define _PAGE_USER	0x004
#define _PAGE_PWT	0x008
#define _PAGE_PCD	0x010
#define _PAGE_ACCESSED	0x020
#define _PAGE_DIRTY	0x040

/*
 * The type of register, physical address and logical address
 */
 
typedef unsigned long reg_t ;
typedef unsigned long phyaddr_t ;
typedef unsigned long logaddr_t ;
/* 
 * pte_t is the type of page table entry
 * pgt_t is the type of page directory entry
 * pgprot_t is the type of property bits
 */
typedef unsigned long pte_t;
typedef unsigned long pgd_t;
typedef unsigned long  pgprot_t;

/* pte_present(x) return the present bit of a page table entry
 *  x is of type pte_t 
 */
#define pte_present(x)	( x & (_PAGE_PRESENT))


#define pte_val(x)	(x)
/*
 * The following only work if pte_present() is true.
 * Undefined behaviour if not..
 */
static inline int pte_read(pte_t pte)		{ return (pte)& _PAGE_USER; }
static inline int pte_exec(pte_t pte)		{ return (pte)& _PAGE_USER; }
static inline int pte_dirty(pte_t pte)		{ return (pte)& _PAGE_DIRTY; }
static inline int pte_young(pte_t pte)		{ return (pte)& _PAGE_ACCESSED; }
static inline int pte_write(pte_t pte)		{ return (pte)& _PAGE_RW; }
/* 
 * mk_pte_phys() creates a page table entry (4 bytes)
 *   physpage: the physical address of a physical frame (4 bytes physical address
 *             which is lowest 12 bits are zero).
 *   pgprot:   the access right bits (the dirty, present bits, etc.)
 * 
 */  
#define __pte(x) ((pte_t)x )
#define __mk_pte(page_nr,pgprot) __pte(((page_nr) << PAGE_SHIFT) | (pgprot))
#define mk_pte_phys(physpage, pgprot)	__mk_pte((physpage) >> PAGE_SHIFT, pgprot)

/* 
 * macros to set insert a page table entry into page table
 *   pteptr, pgdptr : the addresses you want to insert the entry to
 */

#define set_pte(pteptr, pteval) ((*pteptr) = pteval)
#define set_pgd(pgdptr, pgdval) ((*pgdptr) = pgdval)

/* 
 * pgd_offset(base,addr) receives base which is the physical address of page directory
 *     table and a linear address addr. The macro yields the address of the entry in 
 *     a page directory that corresponds
 *     to the address addr. 
 *
 * pgd_index(address) receives a linear address and return the index (p1) to the
 *    page directory table
 */
#define pgd_index(address) ((address >> PGDIR_SHIFT) & (PTRS_PER_PGD-1))
#define __pgd_offset(address) pgd_index(address)
#define pgd_offset(mm, address) (pte_t *)((mm)+pgd_index(address)*4)

/* 
 * pte_offset(pgd_offset,addr) receives a pgd_offset which is an address of a 
 *     page table entry in page directory table and a linear address addr. 
 *     The macro yields the address of the entry corresponding to addr
 *    in the page table referenced by pte.
 * 
 */
 
#define _pte_offset(address) \
		((address >> PAGE_SHIFT) & (PTRS_PER_PTE - 1))
#define pte_offset(pgt_offset, address) \
    ((pte_t *) (((*pgt_offset)& PAGE_MASK)+ _pte_offset(address)*4))
/*
 * clear_page(page) receives a starting address of a frame and clear the frame
 */
#define clear_page(page)	memset((void *)(page), 0, PAGE_SIZE)